import * as d3 from 'd3'
import RadialMenu from './menu'

// 求两点间的距离
function getDis(s, t) {
  return Math.sqrt((s.x - t.x) * (s.x - t.x) + (s.y - t.y) * (s.y - t.y)) || 100
}

// 求元素移动到目标位置所需要的 transform 属性值
function getTransform(source, target, _dis) {
  let r
  if (target.x > source.x) {
    if (target.y > source.y) {
      r = Math.asin((target.y - source.y) / _dis)
    } else {
      r = Math.asin((source.y - target.y) / _dis)
      r = -r
    }
  } else {
    if (target.y > source.y) {
      r = Math.asin((target.y - source.y) / _dis)
      r = Math.PI - r
    } else {
      r = Math.asin((source.y - target.y) / _dis)
      r -= Math.PI
    }
  }
  r = r * (180 / Math.PI)
  return 'translate(' + source.x + ',' + source.y + ')rotate(' + r + ')'
}

// 使文字折行
function textWrap(text, width) {
  text.each(function () {
    const text = d3.select(this),
      words = text.text().split('').reverse(),
      lineHeight = 1.2, // ems
      dy = 0,
      x = text.attr('x')

    let y = text.attr('y')

    if (words.length > 4 && words.length <= 8) {
      y = +y - 4
    } else if (words.length > 8 && words.length <= 10) {
      y = +y - 8
    } else if (words.length > 10) {
      y = +y - 12
    }

    let word,
      line = [],
      lineNumber = 0,
      tspan = text
        .text(null)
        .append('tspan')
        .attr('x', x)
        .attr('y', y)
        .attr('dy', dy + 'em')

    while ((word = words.pop())) {
      line.push(word)
      tspan.text(line.join(' '))
      if (tspan.node().getComputedTextLength() > width) {
        line.pop()
        tspan.text(line.join(' '))
        line = [word]
        tspan = text
          .append('tspan')
          .attr('x', x)
          .attr('y', y)
          .attr('dy', ++lineNumber * lineHeight + dy + 'em')
          .text(word)
      }
    }
  })
}

/*
 *求圆内接正多边形的顶点串，形如x1,y1,x2,y2,...xn,yn
 *cx,cy为圆心，r为半径，n为边数，正多边形底边与X轴平行
 */
function polygonPoints(cx, cy, r, n) {
  const alpha = (2 * Math.PI) / n
  let a = Math.PI / 2 + alpha / 2
  let points = ''
  for (let i = 0; i < n; i++) {
    x = cx + r * Math.cos(a)
    y = cy + r * Math.sin(a)
    points += x + ',' + y + ','
    a += alpha
  }
  return points.substring(0, points.length - 1)
}

// 默认配置
const defaultConfig = {
  nodes: [], // 节点数组
  links: [], // 线数组
  isHighLight: true, // 是否启动 鼠标 hover 到节点上高亮与节点有关的节点，其他无关节点透明的功能
  isScale: true, // 是否启用缩放平移zoom功能
  scaleExtent: [0.6, 1.5], // 缩放的比例尺
  chargeStrength: 300, // 万有引力
  collide: 80, // 碰撞力的大小 （节点之间的间距）
  alphaDecay: 0.1, // 控制力学模拟衰减率
  r: 36, // 圈圈的半径 [30 - 45]
  nodeColor: 'skyblue', // 圈圈节点背景颜色
  fontColor: '#2c3e50', // 圈圈内文字的颜色
  linkSrc: 30, // 划线时候的弧度
  linkColor: 'gray', // 链接线默认的颜色
  strokeColor: 'rgba(255,255,255,0.4)', // 圈圈外围包裹的颜色
  strokeWidth: 6, // 圈圈外围包裹的宽度
  colorList: {
    default: 'skyblue',
    System: '#58b2dc',
    Property: '#ffb11b',
    Subject: '#42b983',
    // Property_property: '#f582ae',
    // Property_produce: '#f3d2c1',
    // Property_manage: '#8bd3dd',
    // Property_service: '#ffd803',
    // Property_use: '#004643',
  },
}

let menu = null

export default class RelationGraph {
  constructor(
    selector,
    data,
    configs = {},
    menuData,
    colorList,
    setCurNode,
    handleLinkClick,
    showBranch
  ) {
    this.menuData = menuData
    this.setCurNode = setCurNode
    this.handleLinkClick = handleLinkClick
    this.showBranch = showBranch

    const mapW = selector.offsetWidth
    const mapH = selector.offsetHeight

    // 画布
    this.graph = d3.select(selector)

    // 合并配置
    this.config = Object.assign(
      {},
      defaultConfig,
      this.transformData(data),
      {
        width: mapW,
        height: mapH,
      },
      configs,
      { colorList }
    )

    // 需要高亮的node和link
    this.dependsNode = []
    this.dependsLink = []

    this.tooltip = d3
      .select('body')
      .append('div')
      .attr('class', 'relation-tooltip')
      .style('opacity', 0)

    this.init()
  }

  // 转换links的source和target
  transformData(data) {
    const nodes = [...data.nodes]
    const links = [...data.links]
    links.forEach((link) => {
      nodes.forEach((node) => {
        if (node.id === link.source) {
          link.source = node
        } else if (node.id === link.target) {
          link.target = node
        }
      })
    })
    const subjects = []
    const objects = []
    const others = []
    nodes.forEach((n) => {
      if (n.subjectType) {
        if (n.subjectType == '1') {
          subjects.push(n)
        } else if (n.subjectType == '2') {
          objects.push(n)
        }
      } else {
        others.push(n)
      }
    })
    return { links, nodes, subjects, objects, others }
  }

  setKey(key) {
    d3.selectAll('.node-text')
      .text((d) => (key && d[key]) || '无')
      // .text((d) => (key && d[key]) || d[d.nodeLabel.toLowerCase() + 'Name'])
      .call(textWrap, this.config.r * 1.8)
  }

  initPatterns() {
    this.patterns && this.patterns.remove()

    // 3.2 添加多个圈圈图片的 <pattern>
    this.patterns = this.defs
      .selectAll('pattern.circle-bg')
      .data(this.config.nodes)
      .enter()
      .append('pattern')
      .attr('class', 'circle-bg')
      .attr('id', (d) => d.id)
      .attr('width', '1')
      .attr('height', '1')

    this.patterns
      .append('rect')
      .attr('width', 2 * this.config.r)
      .attr('height', 2 * this.config.r)
      .attr(
        'fill',
        (d) =>
          (d && this.config.colorList[d._label_key || 'default']) ||
          this.config.nodeColor
      )

    this.patterns
      .append('text')
      .attr('class', 'node-text')
      .attr('x', this.config.r)
      .attr('y', this.config.r) // edit
      .attr('text-anchor', 'middle')
      .attr('dominant-baseline', 'middle')
      .attr('fill', '#000')
      .style('font-size', 12)
      .style('font-weight', 'normal')
      .text((d) => d[d.nodeLabel.toLowerCase() + 'Name'])
  }

  initLinks() {
    const self = this
    // 5.关系图添加线
    this.edges && this.edges.remove()
    // 5.1  每条线是个容器，有线 和一个装文字的容器
    this.edges = this.relMap_g
      .selectAll('g.edge')
      .data(this.config.links)
      .enter()
      .append('g')
      .attr('class', 'edge')
      .on('mouseover', function (e, d) {
        if (self.config.isHighLight) {
          self.highlightLinks(d)
        }
        d3.select(this).selectAll('path.links').attr('stroke-width', 2)
        d3.select(this).selectAll('.rect_g text').style('font-weight', 'bold')
      })
      .on('mouseout', function () {
        if (self.config.isHighLight) {
          self.highlightLinks(null)
        }
        d3.select(this).selectAll('path.links').attr('stroke-width', 1)
        d3.select(this).selectAll('.rect_g text').style('font-weight', 'normal')
      })
      .on('click', function (e, d) {
        console.log('线click')
        self.handleLinkClick(d)
      })
      .attr('fill', (d) => d.color || this.config.linkColor)

    // 5.2 添加线
    this.links = this.edges
      .append('path')
      .attr('class', 'links')
      .attr(
        'd',
        (d) => `M${this.config.linkSrc},0 L${getDis(d.source, d.target)},0`
      )
      .style('marker-end', 'url(#marker)')
      .attr('stroke', (d) => d.color || this.config.linkColor)

    // 5.3 添加关系文字的容器
    this.rect_g = this.edges.append('g').attr('class', 'rect_g')

    // 5.4 添加rect
    this.rects = this.rect_g
      .append('rect')
      .attr('x', 40)
      .attr('y', -10)
      .attr('width', 50)
      .attr('height', 20)
      .attr('fill', '#f9fbfd')
      .attr('stroke', 'transparent')

    // 5.5 文本标签  坐标（x,y）代表 文本的左下角的点
    this.texts = this.rect_g
      .append('text')
      .attr('x', 40)
      .attr('y', 5)
      .attr('text-anchor', 'middle') // <text>文本中轴对齐方式居中  start | middle | end
      .style('font-size', 12)
      .text((d) => d.name)
  }

  initCircles(circleWrapper) {
    const self = this

    // 给圈圈节点添加g便于后面添加menu
    if (circleWrapper) {
      this.circlesWrapper = circleWrapper
    } else {
      this.circlesWrapper = this.relMap_g
        .selectAll('g.circle-wrapper')
        .data(this.config.nodes)
        .enter()
        .append('g')
        .attr('class', 'circle-wrapper')
        .raise()
    }

    // 6.关系图添加用于显示圈圈的节点
    this.circles && this.circles.remove()
    this.circles = this.circlesWrapper
      .append('circle')
      .attr('r', this.config.r)
      .attr('class', 'node')
      .style('cursor', 'pointer')
      .attr('fill', (d) => `url(#${d.id})`)
      .attr('stroke', this.config.strokeColor)
      .attr('stroke-width', this.config.strokeWidth)
      .on('mouseover', function (e, d) {
        d3.select(this).attr(
          'stroke-width',
          self.config.strokeWidth == 0 ? 3 : 1.5 * self.config.strokeWidth
        )
        d3.select(this).attr('stroke', 'rgba(0,0,0,0.1)')
        // d3.select(this).attr('stroke', d.strokeColor || self.config.strokeColor)
        if (self.config.isHighLight) {
          self.highlightObject(d)
        }
        // self.tooltip
        //   .html(d[d.nodeLabel.toLowerCase() + 'Name'])
        //   .style('left', e.pageX + 15 + 'px')
        //   .style('top', e.pageY + 15 + 'px')
        //   .style('opacity', 1)
      })
      .on('mouseout', function (e, d) {
        d3.select(this).attr('stroke-width', self.config.strokeWidth)
        d3.select(this).attr('stroke', d.strokeColor || self.config.strokeColor)
        if (self.config.isHighLight) {
          self.highlightObject(null)
        }
        // self.tooltip.style('opacity', 0)
      })
      .on('click', function (e, d) {
        //阻止事件冒泡  阻止事件默认行为
        e.stopPropagation ? e.stopPropagation() : (e.cancelBubble = true)
        e.preventDefault ? e.preventDefault() : (e.returnValue = false)
        // self.tooltip.style('opacity', 0)
        if (menu && menu.curNodeData == d) {
          self.closeMenu(true)
        } else {
          self.closeMenu(true)
          if (d.nodeLabel === 'Subject') {
            self.openMenu(this, d)
          } else {
            self.openMenu(this, d, ['del', 'branch'])
          }
        }
      })
      .on('contextmenu', function (e) {
        // 取消鼠标右键菜单默认行为
        e.cancelBubble = true
        e.returnValue = false
      })
      // 应用 自定义的 拖拽事件
      .call(
        d3
          .drag()
          .on('start', (e, d) => {
            e.sourceEvent.stopPropagation()
            // restart()方法重新启动模拟器的内部计时器并返回模拟器。
            // 与simulation.alphaTarget或simulation.alpha一起使用时，此方法可用于在交互
            // 过程中进行“重新加热”模拟，例如在拖动节点时，在simulation.stop暂停之后恢复模拟。
            // 当前alpha值为0，需设置alphaTarget让节点动起来
            if (!e.active) this.simulation.alphaTarget(0.3).restart()
            d.fx = d.x
            d.fy = d.y
          })
          .on('drag', (e, d) => {
            // d.fx属性- 节点的固定x位置
            // 在每次tick结束时，d.x被重置为d.fx ，并将节点 d.vx设置为零
            // 要取消节点，请将节点 .fx和节点 .fy设置为空，或删除这些属性。
            d.fx = e.x
            d.fy = e.y
          })
          .on('end', (e, d) => {
            // 让alpha目标值值恢复为默认值0,停止力模型
            if (!e.active) this.simulation.alphaTarget(0)
            d.fx = e.x
            d.fy = e.y
          })
      )

    // 文字折行
    this.SVG.selectAll('text.node-text').call(textWrap, this.config.r * 1.8)
  }

  // 创建力学模拟器
  initSimulation() {
    // 1. 创建一个力学模拟器
    this.simulation = d3
      .forceSimulation(this.config.nodes)
      // simulation.force(name,[force])函数，添加某种力
      .force('link', d3.forceLink(this.config.links))
      // 万有引力
      .force('charge', d3.forceManyBody().strength(this.config.chargeStrength))
      // d3.forceCenter()用指定的x坐标和y坐标创建一个新的居中力。
      .force(
        'center',
        d3.forceCenter(this.config.width / 2, this.config.height / 2)
      )
      // 碰撞作用力，为节点指定一个radius区域来防止节点重叠，设置碰撞力的强度，范围[0,1], 默认为0.7。
      // 设置迭代次数，默认为1，迭代次数越多最终的布局效果越好，但是计算复杂度更高
      .force(
        'collide',
        d3.forceCollide(this.config.collide).strength(0.1).iterations(10)
      )
      // 在计时器的每一帧中，仿真的alpha系数会不断削减,当alpha到达一个系数时，仿真将会停止，也就是alpha的目标系数alphaTarget，该值区间为[0,1]. 默认为0，
      // 控制力学模拟衰减率，[0-1] ,设为0则不停止 ， 默认0.0228，直到0.001
      .alphaDecay(this.config.alphaDecay)
      // 监听事件 ，tick|end ，例如监听 tick 滴答事件
      .on('tick', () => this.ticked())
  }

  update(data) {
    const { nodes, links } = this.transformData(data)

    this.config.nodes = [...nodes]
    this.config.links = [...links]
    console.log('update', this.config)

    this.initPatterns()

    this.initLinks()

    const updateNodes = this.relMap_g
      .selectAll('g.circle-wrapper')
      .data(this.config.nodes)
    updateNodes.exit().remove()
    this.initCircles(
      updateNodes
        .enter()
        .append('g')
        .attr('class', 'circle-wrapper')
        .merge(updateNodes)
        .raise()
    )

    this.simulation
      .nodes(this.config.nodes)
      .force('link', d3.forceLink(this.config.links))
      // .force('charge', null)
      .alpha(1)
      .restart()
  }

  openMenu(self, d, types) {
    this.clearFixedPosition()
    this.setCurNode(d)
    d.fx = d.x
    d.fy = d.y
    this.curActiveNode = d
    menu = new RadialMenu().radius(38).thickness(30).appendTo(self.parentNode)
    if (!types) {
      menu.show(this.menuData, d)
    } else {
      menu.show(
        this.menuData.filter((e) => types.indexOf(e.key) >= 0),
        d
      )
    }
    this.showBranch(d)
    if (this.curHighlightKey) {
      this.setHighlights(this.curHighlightKey)
    }
  }
  clearFixedPosition() {
    this.curActiveNode && (this.curActiveNode.fx = null)
    this.curActiveNode && (this.curActiveNode.fy = null)
  }
  closeMenu(needClearFixedPositionNow) {
    if (menu) {
      menu.hide()
      menu = null
    }
    if (needClearFixedPositionNow) {
      this.clearFixedPosition()
    }
    // else {
    //   setTimeout(() => this.clearFixedPosition(), 1000)
    // }
  }

  init() {
    // 2.创建svg标签
    this.SVG = this.graph
      .append('svg')
      .attr('width', this.config.width)
      .attr('height', this.config.height)
      .call(
        d3
          .zoom()
          .scaleExtent(this.config.scaleExtent)
          .on('zoom', (e) => {
            if (this.config.isScale) {
              this.relMap_g.attr('transform', e.transform)
            }
          })
      )
      .on('dblclick.zoom', null) // 取消双击放大
      .on('click', () => this.closeMenu())

    // 3.defs  <defs>标签的内容不会显示，只有调用的时候才显示
    this.defs = this.SVG.append('defs')

    // 3.1 添加箭头
    const border = 7.5
    //const border = this.config.strokeWidth == 0 ? 7.5 : 1.8 * this.config.strokeWidth
    this.marker = this.defs
      .append('marker')
      .attr('id', 'marker')
      .attr('markerWidth', 10) //marker视窗的宽
      .attr('markerHeight', 10) //marker视窗的高
      .attr('refX', this.config.r + border) //refX和refY，指的是图形元素和marker连接的位置坐标
      .attr('refY', 4)
      .attr('orient', 'auto') //orient="auto"设置箭头的方向为自动适应线条的方向
      .attr('markerUnits', 'userSpaceOnUse') //marker是否进行缩放 ,默认值是strokeWidth,会缩放
      .append('path')
      .attr('d', 'M 0 0 8 4 0 8Z') //箭头的路径 从 （0,0） 到 （8,4） 到（0,8）
      .attr('fill', this.config.linkColor)

    this.initSimulation()

    this.initPatterns()

    // 4.放关系图的容器
    this.relMap_g = this.SVG.append('g')
      .attr('class', 'relMap_g')
      .attr('width', this.config.width)
      .attr('height', this.config.height)

    this.initLinks()

    this.initCircles()
  }

  ticked() {
    // 7.1 修改每条容器edge的位置
    this.edges.attr('transform', (d) =>
      getTransform(d.source, d.target, getDis(d.source, d.target))
    )

    // 7.2 修改每条线link位置
    this.links.attr(
      'd',
      (d) => `M${this.config.linkSrc},0 L${getDis(d.source, d.target)},0`
    )

    // 7.3 修改线中关系文字text的位置 及 文字的反正
    this.texts
      .attr('x', function (d) {
        // // 7.3.1 根据字的长度来更新兄弟元素 rect 的宽度
        // const { width } = d3.select(this).node().getBBox()

        // 7.3.2 更新 text 的位置
        return getDis(d.source, d.target) / 2
      })
      .attr('transform', function (d) {
        // 7.3.3 更新文本反正
        if (d.target.x < d.source.x) {
          const x = getDis(d.source, d.target) / 2
          return 'rotate(180 ' + x + ' ' + 0 + ')'
        } else {
          return 'rotate(0)'
        }
      })

    // 7.4 修改线中装文本矩形rect的位置
    this.rects.attr('x', function (d) {
      return getDis(d.source, d.target) / 2 - d3.select(this).attr('width') / 2
    }) // x 坐标为两点中心距离减去自身长度一半

    // 5.修改节点的位置
    this.circles.attr('cx', (d) => d.x).attr('cy', (d) => d.y)

    // 让menu随圆圈移动
    this.circles.each(function () {
      const menuEle = d3.select(this.nextSibling)
      if (!menuEle.empty()) {
        const self = d3.select(this)
        const cx = self.attr('cx')
        const cy = self.attr('cy')
        menuEle.attr('transform', `translate(${cx},${cy})`)
      }
    })
  }

  // 高亮元素及其相关的元素
  highlightObject(obj) {
    if (obj) {
      const objIndex = obj.index
      this.dependsNode = this.dependsNode.concat([objIndex])
      this.dependsLink = []

      this.config.links.forEach((link) => {
        if (objIndex == link.source.index) {
          this.dependsNode.push(link.target.index)
          this.dependsLink.push(link.index)
        } else if (objIndex == link.target.index) {
          this.dependsNode.push(link.source.index)
          this.dependsLink.push(link.index)
        }
      })

      /** 二级节点也要展示 不展示可以删除 */
      const secondNodes = this.dependsNode.filter((e) => e !== objIndex)
      this.config.links.forEach((link) => {
        if (secondNodes.indexOf(link.source.index) >= 0) {
          this.dependsNode.push(link.target.index)
          this.dependsLink.push(link.index)
        } else if (secondNodes.indexOf(link.target.index) >= 0) {
          this.dependsNode.push(link.source.index)
          this.dependsLink.push(link.index)
        }
      })

      this.dependsNode = Array.from(new Set(this.dependsNode))
      this.dependsLink = Array.from(new Set(this.dependsLink))

      // 隐藏节点
      this.SVG.selectAll('circle')
        .filter((d) => this.dependsNode.indexOf(d.index) == -1)
        .transition()
        .style('opacity', 0.2)

      // 隐藏线
      this.SVG.selectAll('.edge')
        .filter((d) => this.dependsLink.indexOf(d.index) == -1)
        .transition()
        .style('opacity', 0.1)
    } else {
      // 取消高亮
      // 恢复隐藏的线
      this.SVG.selectAll('circle').transition().style('opacity', 1)

      // 恢复隐藏的线
      this.SVG.selectAll('.edge').transition().style('opacity', 1)

      this.dependsNode = []
      this.dependsLink = []
    }
  }

  highlightLinks(obj) {
    if (obj) {
      // 隐藏节点
      this.SVG.selectAll('circle')
        .filter((d) => {
          const sourceIndex = this.config.nodes.findIndex(
            (e) => e.id == obj.source.id
          )
          const targetIndex = this.config.nodes.findIndex(
            (e) => e.id == obj.target.id
          )
          return d.index != sourceIndex && d.index != targetIndex
        })
        .transition()
        .style('opacity', 0.2)

      // 隐藏线
      this.SVG.selectAll('.edge')
        .filter((d) => d.id != obj.id)
        .transition()
        .style('opacity', 0.1)
    } else {
      this.SVG.selectAll('circle').transition().style('opacity', 1)
      this.SVG.selectAll('.edge').transition().style('opacity', 1)
    }
  }

  setHighlights(key) {
    this.curHighlightKey = key
    this.config.isHighLight = false
    this.SVG.selectAll('circle').transition().style('opacity', 1)
    this.SVG.selectAll('.edge').transition().style('opacity', 1)
    this.dependsNode = []
    this.dependsLink = []
    if (!key || this.curActiveNode.nodeLabel !== 'Subject') {
      this.curHighlightKey = null
      this.config.isHighLight = true
      return false
    }

    this.highlightObject(this.curActiveNode)

    this.dependsNode = [this.curActiveNode.index]
    this.config.links.forEach((link) => {
      const index = this.dependsLink.indexOf(link.index)
      if (index < 0) return
      if (link.target._label_key != key && link.source._label_key != key) {
        this.dependsLink.splice(index, 1)
      } else if (
        link.target._label_key == key &&
        link.source.id != this.curActiveNode.id
      ) {
        this.dependsLink.splice(index, 1)
      } else {
        this.dependsNode.push(link.target.index)
        this.dependsNode.push(link.source.index)
      }
    })

    // 隐藏节点
    this.SVG.selectAll('circle')
      .filter((d) => this.dependsNode.indexOf(d.index) < 0)
      .transition()
      .style('opacity', 0.2)

    // 隐藏线
    this.SVG.selectAll('.edge')
      .filter((d) => this.dependsLink.indexOf(d.index) < 0)
      .transition()
      .style('opacity', 0.2)

    return true
  }
}
