import MyMap from '../my-map'
import {
  CallBack,
  MapConfig,
  Listeners,
  ControlOptions,
  Location,
  FocusOptions,
  Controls,
  SMapConfig,
  Layers,
  LayerOption,
  ClickCallBack,
} from '../types'
import { toLines, toPoints } from './util'

export default class S_Map extends MyMap {
  constructor(config: MapConfig<SMapConfig>) {
    if (!config.netType) {
      throw new Error('SMap需要设置netType参数！')
    }
    new SMap.Network().setNet(config.netType)
    const instance = new SMap.Map(config.el, {
      appKey: config.appKey,
      viewMode: config.mode,
      center: config.center,
      zoom: config.zoom,
      zooms: config.zooms,
      pitch: config.pitch,
      mapStyle: config.style,
      showBuildingBlock: config.showBuildingBlock,
      rotateEnable: config.rotateEnable,
    })
    super(instance)
    this._setListeners()
    this._setControls()
    this._setLayers()
    this.on('load', this._clearFooter)
  }

  /** 清除地图自带的脚注 */
  private _clearFooter() {
    const footer = document.querySelector(
      '.esri-ui-manual-container>.esri-component',
    )
    footer && ((footer as HTMLElement).style.display = 'none')
  }

  /** 设置监听事件 */
  private _setListeners() {
    const _listeners: Partial<Listeners> = {
      load: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.maploaded, cb)
      },
      zoom: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.zoomchanged, cb)
      },
      move: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.centerchanged, cb)
      },
      blur: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.blur, cb)
      },
      focus: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.focus, cb)
      },
      drag: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.drag, cb)
      },
      resize: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.resize, cb)
      },
      click: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.click, cb)
      },
      dblclick: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.doubleclick, cb)
      },
      mousewheel: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.mousewheel, cb)
      },
    }
    this._listeners = Object.assign(this._listeners, _listeners)
  }

  /** 特殊的可能会带自定义数据的事件 */
  private _addEvent(event: unknown, cb: ClickCallBack) {
    this.map.on(event, (view: any, e: any) => {
      const arg = {
        type: e.type,
        center: [e.mapPoint?.longitude, e.mapPoint?.latitude],
        event: e.native,
        clientX: e.x,
        clientY: e.y,
        x: e?.mapPoint?.x,
        y: e?.mapPoint?.y,
        originData: e,
      }
      view.hitTest(e).then((res: any) => {
        cb(arg, res?.results?.[0]?.graphic?.attributes || null)
      })
    })
  }

  /** 设置控制器 */
  private _setControls() {
    const _controls: Partial<Controls> = {
      home: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Home({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      compass: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Compass({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      zoom: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Zoom({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      fullScreen: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Fullscreen({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      layerList: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.LayerListControl({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      measureLine: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.MeasureLine({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      measureArea: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.MeasureArea({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      basemapToggle: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.BasemapToggle({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      underguroundSwitch: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.UndergroundSwitch({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      bMapGallery: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.BMapGallery({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      bMapGalleryexpand: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.BMapGalleryExpand({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
    }
    this._controls = Object.assign(this._controls, _controls)
  }

  /** 设置覆盖物 */
  private _setLayers() {
    const _markers: Partial<Layers> = {
      point: (data: LayerOption) => {
        const layer = new SMap.OverlayGroup(toPoints(data), {})
        this.map.add(layer)
        return layer
      },
      line: (data: LayerOption) => {
        const layer = new SMap.OverlayGroup(toLines(data), {})
        this.map.add(layer)
        return layer
      },
    }
    this._markers = Object.assign(this._markers, _markers)
  }

  zoomIn() {
    this.map.zoomIn()
  }
  zoomOut() {
    this.map.zoomOut()
  }
  zoomTo(level: number) {
    this.map.setZoom(level)
  }
  focus(location: Location, options?: FocusOptions) {
    const level = options?.zoom || this.map.getZoom()
    this.map.setZoomAndCenter(level, location)
  }
  remove(layer: unknown) {
    layer && this.map.remove(layer)
  }
}
