import MyMap from './my-map'
import {
  CallBack,
  MapConfig,
  Listeners,
  ControlOptions,
  Location,
  FocusOptions,
  Controls,
  SMapConfig,
  Markers,
  PointData,
  ClickCallBack,
} from './types'

declare const SMap: {
  Map: any
  MapEvent: any
  Size: any
  Icon: any
  Label: any
  Marker: any
  OverlayGroup: any
  Network: any
  Home: any
  Zoom: any
  Compass: any
  Fullscreen: any
  LayerListControl: any
  MeasureLine: any
  MeasureArea: any
  BasemapToggle: any
  UndergroundSwitch: any
  BMapGallery: any
  BMapGalleryExpand: any
}
declare const Plugins: {
  MaskBoundary: any
}

export default class S_Map extends MyMap {
  constructor(config: MapConfig<SMapConfig>) {
    if (!config.netType) {
      throw new Error('SMap需要设置netType参数！')
    }
    new SMap.Network().setNet(config.netType)
    const instance = new SMap.Map(config.el, {
      appKey: config.appKey,
      viewMode: config.mode,
      center: config.center,
      zoom: config.zoom,
      zooms: config.zooms,
      pitch: config.pitch,
      mapStyle: config.style,
      showBuildingBlock: config.showBuildingBlock,
      rotateEnable: config.rotateEnable,
    })
    super(instance)
    this._setListeners()
    this._setControls()
    this._setMarkers()
    this.on('load', this._clearFooter)
  }

  /** 清除地图自带的脚注 */
  private _clearFooter() {
    const footer = document.querySelector(
      '.esri-ui-manual-container>.esri-component',
    )
    footer && ((footer as HTMLElement).style.display = 'none')
  }

  /** 设置监听事件 */
  private _setListeners() {
    const _listeners: Partial<Listeners> = {
      load: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.maploaded, cb)
      },
      zoom: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.zoomchanged, cb)
      },
      move: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.centerchanged, cb)
      },
      blur: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.blur, cb)
      },
      focus: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.focus, cb)
      },
      drag: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.drag, cb)
      },
      resize: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.resize, cb)
      },
      click: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.click, cb)
      },
      dblclick: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.doubleclick, cb)
      },
      mousewheel: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.mousewheel, cb)
      },
    }
    this._listeners = Object.assign(this._listeners, _listeners)
  }

  /** 特殊的可能会带自定义数据的事件 */
  private _addEvent(event: unknown, cb: ClickCallBack) {
    this.map.on(event, (view: any, e: any) => {
      const arg = {
        type: e.type,
        center: [e.mapPoint?.longitude, e.mapPoint?.latitude],
        event: e.native,
        clientX: e.x,
        clientY: e.y,
        x: e?.mapPoint?.x,
        y: e?.mapPoint?.y,
        originData: e,
      }
      view.hitTest(e).then((res: any) => {
        cb(arg, res?.results?.[0]?.graphic?.attributes || null)
      })
    })
  }

  /** 设置控制器 */
  private _setControls() {
    const _controls: Partial<Controls> = {
      home: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Home({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      compass: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Compass({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      zoom: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Zoom({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      fullScreen: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Fullscreen({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      layerList: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.LayerListControl({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      measureLine: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.MeasureLine({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      measureArea: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.MeasureArea({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      basemapToggle: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.BasemapToggle({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      underguroundSwitch: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.UndergroundSwitch({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      bMapGallery: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.BMapGallery({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      bMapGalleryexpand: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.BMapGalleryExpand({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
    }
    this._controls = Object.assign(this._controls, _controls)
  }

  /** 转化点状覆盖物 */
  private _toPoints(options: PointData) {
    const { data, size = 20, icon, labelKey, labelOptions } = options

    return data.map((item) => {
      const iconSize = Array.isArray(size)
        ? new SMap.Size(...size)
        : new SMap.Size(size, size)
      const x = item.X || item.x || item.gpsx
      const y = item.Y || item.y || item.gpsy
      if ((x !== 0 && !x) || (y !== 0 && !y)) {
        console.error(`add point: 非法的坐标[${x}, ${y}] 存在于数据: ${item}`)
      }
      const z = item.Z || item.z || item.gpsz || 0
      const result: any = {
        icon: {
          size: iconSize,
          image: item.icon || icon,
        },
        attributes: { ...item },
        position: z ? [x, y, z] : [x, y],
      }
      if (labelKey) {
        result.label = new SMap.Label({
          text: item[labelKey] + '',
          size: labelOptions?.size,
          color: labelOptions?.color,
          xoffset: labelOptions?.offset?.[0],
          yoffset: labelOptions?.offset?.[1],
          zoffset: labelOptions?.offset?.[2],
          verticalAlignment: 'middle',
          horizontalAlignment: 'center',
        })
      }
      return new SMap.Marker(result)
    })
  }

  /** 设置覆盖物方法 */
  private _setMarkers() {
    const _markers: Partial<Markers> = {
      point: (data: PointData) => {
        const layer = new SMap.OverlayGroup(this._toPoints(data), {})
        this.instance.add(layer)
        return layer
      },
    }
    this._markers = Object.assign(this._markers, _markers)
  }

  zoomIn() {
    this.map.zoomIn()
  }
  zoomOut() {
    this.map.zoomOut()
  }
  zoomTo(level: number) {
    this.map.setZoom(level)
  }
  focus(location: Location, options?: FocusOptions) {
    const level = options?.zoom || this.map.getZoom()
    this.map.setZoomAndCenter(level, location)
  }
  remove(layer: unknown) {
    this.map.remove(layer)
  }
}
