import MyMap from './my-map'
import {
  CallBack,
  MapConfig,
  Listeners,
  ControlOptions,
  ZoomOptions,
  FocusOptions,
  Location,
  ClickCallBack,
  Controls,
  AIMapConfig,
  Markers,
  PointData,
} from './types'

declare const aimap: {
  Map: any
  accessToken: string
  baseApiUrl: string
  NavigationControl: any
  CompassControl: any
  ScaleControl: any
  MassMarkerLayer: any
}

export default class AI_Map extends MyMap {
  private _clickCb: ClickCallBack | null = null
  constructor(config: MapConfig<AIMapConfig>) {
    if (!config.appKey) {
      throw new Error('AIMap需要设置appKey参数！')
    }
    aimap.accessToken = config.appKey
    if (!config.baseApiUrl) {
      throw new Error('AIMap需要设置baseApiUrl参数！')
    }
    aimap.baseApiUrl = config.baseApiUrl

    const instance = new aimap.Map({
      container: config.el,
      center: config.center,
      zoom: config.zoom,
      minZoom: config.zooms?.[0],
      maxZoom: config.zooms?.[1],
      pitch: config.pitch || 0,
      minPitch: config.pitchs?.[0],
      maxPitch: config.pitchs?.[1],
      bearing: config.bearing || 0,
      style: config.style,
      localIdeographFontFamily: config.family,
    })
    super(instance)
    this._setListeners()
    this._setControls()
    this._setMarkers()
  }

  /** 设置监听事件 */
  private _setListeners() {
    const _listeners: Partial<Listeners> = {
      load: (cb: CallBack) => {
        this.map.on('load', cb)
      },
      zoom: (cb: CallBack) => {
        this.map.on('zoom', cb)
      },
      move: (cb: CallBack) => {
        this.map.on('move', cb)
      },
      blur: (cb: CallBack) => {
        this.map.on('blur', cb)
      },
      focus: (cb: CallBack) => {
        this.map.on('focus', cb)
      },
      drag: (cb: CallBack) => {
        this.map.on('drag', cb)
      },
      resize: (cb: CallBack) => {
        this.map.on('resize', cb)
      },
      click: (cb: ClickCallBack) => {
        this._clickCb = cb
        this.map.on('click', (e: any) => {
          const arg = {
            type: e.type,
            center: [e.lngLat.lng, e.lngLat.lat],
            event: e.originalEvent,
            clientX: e.point.x,
            clientY: e.point.y,
            originData: e,
          }
          cb(arg)
        })
      },
    }
    this._listeners = Object.assign(this._listeners, _listeners)
  }

  /** 设置控制器 */
  private _setControls() {
    const _controls: Partial<Controls> = {
      compass: (options?: ControlOptions) => {
        this.map.addControl(new aimap.CompassControl(), options?.position)
      },
      zoom: (options?: ControlOptions) => {
        this.map.addControl(
          new aimap.NavigationControl({
            showZoom: options?.show,
            showCompass: false,
            visualizePitch: false,
          }),
          options?.position,
        )
      },
      scale: (options?: ControlOptions) => {
        this.map.addControl(
          new aimap.ScaleControl({
            maxWidth: options?.maxWidth || 80,
            unit: options?.unit || 'metric',
          }),
          options?.position,
        )
      },
    }
    this._controls = Object.assign(this._controls, _controls)
  }

  /** 转化点状覆盖物 */
  private _toPoints(options: PointData) {
    const {
      data,
      size = 20,
      icon,
      labelKey,
      spatialReference = 'gcj02',
      labelOptions,
    } = options
    const [horizontal, vertical] = labelOptions?.offset || []
    return {
      map: this.map,
      spatialReference: spatialReference,
      data: data.map((item, i: number) => {
        const x = item.X || item.x || item.gpsx
        const y = item.Y || item.y || item.gpsy
        if ((x !== 0 && !x) || (y !== 0 && !y)) {
          console.error(`add point: 非法的坐标[${x}, ${y}] 存在于数据: ${item}`)
        }
        const z = item.Z || item.z || item.gpsz || 0
        return {
          ...item,
          id: item.id || i,
          icon: item.icon ? `icon${i}` : 'icon',
          spatialReference: spatialReference || item.spatialReference,
          coordinates: z ? [x, y, z] : [x, y],
        }
      }),
      images: data.map((item: any, i: number) => {
        return {
          id: item.icon ? `icon${i}` : 'icon',
          url: item.icon || icon,
        }
      }),
      style: {
        'text-field': `{${labelKey}}`,
        'text-color': labelOptions?.color,
        // 偏移相比测绘院地图大概小了10倍，且竖直偏移相反
        'text-offset': [
          horizontal ? horizontal / 10 : 0,
          vertical ? vertical / -10 : 0,
        ],
        'icon-anchor': 'center',
        'icon-image': ['get', 'icon'],
        // size相比测绘院地图大概小了50倍
        'icon-size': Array.isArray(size) ? size[0] / 50 : size / 50,
      },
    }
  }

  /** 设置覆盖物方法 */
  private _setMarkers() {
    const _markers: Partial<Markers> = {
      point: (data: PointData) => {
        const layer = new aimap.MassMarkerLayer(this._toPoints(data))
        // 维智的覆盖物点击事件需特别定义
        layer.on('click', (e: any) => {
          const arg = {
            type: e.type,
            center: [e.lngLat.lng, e.lngLat.lat],
            event: e.originalEvent,
            clientX: e.point.x,
            clientY: e.point.y,
            originData: e,
          }
          // 点击覆盖物获取自定义数据
          this._clickCb && this._clickCb(arg, e?.features?.[0]?.properties)
        })
        return layer
      },
    }
    this._markers = Object.assign(this._markers, _markers)
  }

  zoomIn(options: ZoomOptions) {
    this.map.zoomIn(options)
  }
  zoomOut(options: ZoomOptions) {
    this.map.zoomOut(options)
  }
  zoomTo(level: number, options: ZoomOptions) {
    this.map.zoomTo(level, options)
  }
  focus(location: Location, options?: FocusOptions) {
    this.map.flyTo({
      center: location,
      ...options,
    })
  }
  remove(layer: any) {
    layer.remove()
  }
}
