import MyMap from './my-map'
import {
  CallBack,
  MapConfig,
  Listeners,
  ControlOptions,
  ZoomOptions,
  FocusOptions,
  Location,
} from './types'

declare const aimap: {
  Map: any
  accessToken: string
  baseApiUrl: string
  NavigationControl: any
  CompassControl: any
  ScaleControl: any
}

export default class AI_Map extends MyMap {
  constructor(config: MapConfig) {
    if (!config.appKey) {
      throw new Error('AIMap需要设置appKey参数！')
    }
    aimap.accessToken = config.appKey
    if (!config.baseApiUrl) {
      throw new Error('AIMap需要设置baseApiUrl参数！')
    }
    aimap.baseApiUrl = config.baseApiUrl

    const instance = new aimap.Map({
      container: config.el,
      center: config.center,
      zoom: config.zoom,
      minZoom: config.zooms?.[0],
      maxZoom: config.zooms?.[1],
      pitch: config.pitch || 0,
      minPitch: config.pitchs?.[0],
      maxPitch: config.pitchs?.[1],
      bearing: config.bearing || 0,
      style: config.style,
      localIdeographFontFamily: config.family,
      spatialReference: 'cgcs2000',
    })
    super(instance)
    this.setListeners()
    this.setControls()
  }

  private setListeners() {
    const _listeners: Partial<Listeners> = {
      load: (cb: CallBack) => {
        this.map.on('load', cb)
      },
      zoom: (cb: CallBack) => {
        this.map.on('zoom', cb)
      },
      move: (cb: CallBack) => {
        this.map.on('move', cb)
      },
      blur: (cb: CallBack) => {
        this.map.on('blur', cb)
      },
      focus: (cb: CallBack) => {
        this.map.on('focus', cb)
      },
      drag: (cb: CallBack) => {
        this.map.on('drag', cb)
      },
      resize: (cb: CallBack) => {
        this.map.on('resize', cb)
      },
      click: (cb: CallBack) => {
        this.map.on('click', cb)
      },
    }
    this._listeners = Object.assign(this._listeners, _listeners)
  }

  private setControls() {
    const _controls = {
      compass: (options: ControlOptions) => {
        this.map.addControl(new aimap.CompassControl(), options.position)
      },
      zoom: (options: ControlOptions) => {
        this.map.addControl(
          new aimap.NavigationControl({
            showZoom: options.show,
            showCompass: false,
            visualizePitch: false,
          }),
          options.position,
        )
      },
      scale: (options: ControlOptions) => {
        this.map.addControl(
          new aimap.ScaleControl({
            maxWidth: options.maxWidth || 80,
            unit: options.unit || 'metric',
          }),
          options.position,
        )
      },
    }
    this._controls = Object.assign(this._controls, _controls)
  }

  zoomIn(options: ZoomOptions) {
    this.map.zoomIn(options)
  }
  zoomOut(options: ZoomOptions) {
    this.map.zoomOut(options)
  }
  zoomTo(level: number, options: ZoomOptions) {
    this.map.zoomTo(level, options)
  }
  focus(location: Location, options?: FocusOptions) {
    this.map.flyTo({
      center: location,
      ...options,
    })
  }
}
