import MyMap from './my-map'
import {
  CallBack,
  MapConfig,
  Listeners,
  ControlOptions,
  Location,
  FocusOptions,
  Controls,
  SMapConfig,
} from './types'

declare const SMap: {
  Map: any
  MapEvent: any
  Size: any
  Icon: any
  Label: any
  Marker: any
  Network: any
  Home: any
  Zoom: any
  Compass: any
  Fullscreen: any
  LayerListControl: any
  MeasureLine: any
  MeasureArea: any
  BasemapToggle: any
  UndergroundSwitch: any
  BMapGallery: any
  BMapGalleryExpand: any
}
declare const Plugins: {
  MaskBoundary: any
}

export default class S_Map extends MyMap {
  constructor(config: MapConfig<SMapConfig>) {
    if (!config.netType) {
      throw new Error('SMap需要设置netType参数！')
    }
    new SMap.Network().setNet(config.netType)
    const instance = new SMap.Map(config.el, {
      appKey: config.appKey,
      viewMode: config.mode,
      center: config.center,
      zoom: config.zoom,
      zooms: config.zooms,
      pitch: config.pitch,
      mapStyle: config.style,
      showBuildingBlock: config.showBuildingBlock,
      rotateEnable: config.rotateEnable,
    })
    super(instance)
    this._setListeners()
    this._setControls()
    this.on('load', this._clearFooter)
  }

  private _clearFooter() {
    const footer = document.querySelector(
      '.esri-ui-manual-container>.esri-component',
    )
    footer && ((footer as HTMLElement).style.display = 'none')
  }

  private _setListeners() {
    const _listeners: Partial<Listeners> = {
      load: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.maploaded, cb)
      },
      zoom: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.zoomchanged, cb)
      },
      move: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.centerchanged, cb)
      },
      blur: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.blur, cb)
      },
      focus: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.focus, cb)
      },
      drag: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.drag, cb)
      },
      resize: (cb: CallBack) => {
        this.map.on(SMap.MapEvent.resize, cb)
      },
      click: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.click, cb)
      },
      dblclick: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.doubleclick, cb)
      },
      mousewheel: (cb: CallBack) => {
        this._addEvent(SMap.MapEvent.mousewheel, cb)
      },
    }
    this._listeners = Object.assign(this._listeners, _listeners)
  }

  private _addEvent(event: unknown, cb: CallBack) {
    this.map.on(event, (view: any, e: any) => {
      const arg = {
        type: e.type,
        center: [e.mapPoint?.longitude, e.mapPoint?.latitude],
        event: e.native,
        clientX: e.x,
        clientY: e.y,
        originData: e,
      }
      view.hitTest(e).then((res: any) => {
        cb(arg, res.results)
      })
    })
  }

  private _setControls() {
    const _controls: Partial<Controls> = {
      home: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Home({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      compass: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Compass({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      zoom: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Zoom({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      fullScreen: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.Fullscreen({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      layerList: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.LayerListControl({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      measureLine: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.MeasureLine({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      measureArea: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.MeasureArea({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      basemapToggle: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.BasemapToggle({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      underguroundSwitch: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.UndergroundSwitch({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      bMapGallery: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.BMapGallery({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
      bMapGalleryexpand: (options?: ControlOptions) => {
        this.map.addControl(
          new SMap.BMapGalleryExpand({
            visible: options?.show,
            position: options?.position,
          }),
        )
      },
    }
    this._controls = Object.assign(this._controls, _controls)
  }

  zoomIn() {
    this.map.zoomIn()
  }
  zoomOut() {
    this.map.zoomOut()
  }
  zoomTo(level: number) {
    this.map.setZoom(level)
  }
  focus(location: Location, options?: FocusOptions) {
    const level = options?.zoom || this.map.getZoom()
    this.map.setZoomAndCenter(level, location)
  }
}
