import * as vscode from "vscode";
import path = require("path");
import fs = require("fs");
const download = require("npm-gitee-lw");

export const getWebViewContent = (filePath: string) => {
  const dirname = path.dirname(filePath);
  const resourcePath = path.join(__dirname, dirname);
  const filename = path.basename(filePath);
  const html = fs.readFileSync(path.join(resourcePath, filename), "utf-8");
  // vscode不支持直接加载本地资源，需要替换成其专有路径格式，这里只是简单的将样式和JS的路径替换
  return html.replace(
    /(<link.+?href="|<script.+?src="|<img.+?src=")(.+?)"/g,
    (m, $1, $2) => {
      return (
        $1 +
        vscode.Uri.file(path.resolve(resourcePath, $2))
          .with({ scheme: "vscode-resource" })
          .toString() +
        '"'
      );
    }
  );
};

export const isDirExist = (path: string) => {
  try {
    fs.accessSync(path, fs.constants.R_OK | fs.constants.W_OK);
    return true;
  } catch (err) {
    return false;
  }
};

export interface FileConfig {
  url: string;
  filePath: string;
  filename: string;
}

export const fetchProject = ({
  url,
  filePath,
  filename,
}: FileConfig): Promise<void> => {
  const projectPath = `${filePath}/${filename}`;
  return new Promise((resolve, reject) => {
    if (isDirExist(projectPath)) {
      vscode.window.showErrorMessage("该路径下已存在同名项目文件夹！");
      return reject();
    }
    download(url, projectPath, { clone: true }, async (err: Error) => {
      if (err) {
        vscode.window.showErrorMessage(err.message);
        throw new Error(err.message);
      }
      vscode.window.showInformationMessage(`'${filename}' 项目创建成功!`);
      await vscode.commands.executeCommand(
        "vscode.openFolder",
        vscode.Uri.file(projectPath),
        true
      );
      return resolve();
    });
  });
};

const packageInfo = ({
  name,
  path,
}: {
  name: string;
  path: string;
}): Promise<any> => {
  return new Promise((resolve, reject) => {
    fs.readFile(`${path}/package.json`, "utf8", (err, data) => {
      if (err) {
        return reject(err);
      }
      const packageJson = JSON.parse(data);
      packageJson.name = name;
      return resolve(JSON.stringify(packageJson, null, 2));
    });
  });
};

export const updatePackageJson = async ({
  filePath,
  filename,
}: FileConfig): Promise<void> => {
  const path = `${filePath}/${filename}`;
  const data = await packageInfo({ path, name: filename });
  return new Promise((resolve, reject) => {
    fs.writeFile(`${path}/package.json`, data, "utf8", (err) => {
      if (err) {
        return reject();
      }
      return resolve();
    });
  });
};
