import { MapConfig, CallBack, BoundaryOption, MarkerOption } from './types'
import Map from './map'

declare const SMap: {
  Map: any
  Network: any
  MapEvent: any
  Size: any
  Icon: any
  Label: any
  Marker: any
}

declare const Plugins: {
  MaskBoundary: any
}

export default class S_Map extends Map {
  constructor(config: MapConfig) {
    new SMap.Network().setNet(config.options.netType || 'internet')
    super(new SMap.Map(config.el, config.options))
  }
  onLoad(cb: CallBack): void {
    this.map.on(SMap.MapEvent.maploaded, cb)
  }
  onZoom(cb: CallBack): void {
    this.map.on(SMap.MapEvent.zoomchanged, cb)
  }
  onMove(cb: CallBack): void {
    this.map.on(SMap.MapEvent.centerchanged, cb)
  }
  private event(event: unknown, cb: CallBack) {
    this.map.on(event, (view: any, eventParamter: unknown) => {
      view.hitTest(eventParamter).then((res: unknown) => {
        cb(res, eventParamter)
      })
    })
  }
  onBlur(cb: CallBack): void {
    this.event(SMap.MapEvent.blur, cb)
  }
  onFocus(cb: CallBack): void {
    this.event(SMap.MapEvent.focus, cb)
  }
  onDrag(cb: CallBack): void {
    this.event(SMap.MapEvent.drag, cb)
  }
  onResize(cb: CallBack): void {
    this.event(SMap.MapEvent.resize, cb)
  }
  onClick(cb: CallBack): void {
    this.event(SMap.MapEvent.click, cb)
  }
  onDblclick(cb: CallBack): void {
    this.event(SMap.MapEvent.doubleclick, cb)
  }
  onMousewheel(cb: CallBack): void {
    this.event(SMap.MapEvent.mousewheel, cb)
  }
  zoomIn(): void {
    this.map.zoomIn()
  }
  zoomOut(): void {
    this.map.zoomOut()
  }
  hideCommunity(): void {
    this.map.hideXQ_Poly()
    this.map.hideMPZ()
    this.map.hideXQ_Point()
  }
  showCommunity(): void {
    this.map.showXQ_Poly()
    this.map.showMPZ()
    this.map.showXQ_Point()
  }
  filterCommunity(ids: unknown[], key = 'sect_id'): void {
    if (ids.length === 0) return
    const valid = ids.map((val: unknown) => {
      return `${key} = '${val}'`
    })
    this.map.setFilter('wg_gis_xq_point', valid.join(' or '))
    this.map.setFilter('wg_gis_mpz', valid.join(' or '))
    this.map.setFilter('wg_gis_xq_poly', valid.join(' or '))
  }
  focus(location: number[], zoom = this.map.getZoom()): void {
    this.map.setZoomAndCenter(zoom, location)
  }
  addBoundary({
    name,
    type,
    weight = 10,
    count = 10,
    color = 'rgba(51,145,255,.6)',
    maskColor = [0, 17, 33, 0.9],
  }: BoundaryOption): unknown {
    if (!name || !type) {
      throw new Error('SMap的addBoundary方法需要name和type参数！')
      return
    }
    const boundary = {
      boundaryType: type,
      boundaryDefinition: `name like '%${name}%'`,
      boundarydistance: weight,
      bounarycount: count,
      boundaryColor: color,
      maskColor: maskColor,
    }
    const Boundary = new Plugins.MaskBoundary(this.map.view)
    Boundary.add(boundary)
    return Boundary
  }
  private transMarkerData(option: MarkerOption): unknown[] {
    const { data = [], size, icon, key, labelKey, color } = option
    return data.map((item: any) => {
      let sizeProp = Array.isArray(size)
        ? new SMap.Size(size[0], size[1])
        : new SMap.Size(size, size)
      if (item.size) {
        sizeProp = Array.isArray(item.size)
          ? new SMap.Size(item.size[0], item.size[1])
          : new SMap.Size(item.size, item.size)
      }
      const result = {
        icon: new SMap.Icon({
          size: sizeProp,
          image: item.icon || icon,
        }),
        attributes: { key, ...item },
        position: [
          item.X || item.x || item.gpsx || item.lon,
          item.Y || item.y || item.gpsy || item.lat,
          item.Z || item.z || item.gpsz || 0,
        ],
        label: new SMap.Label({
          text: labelKey ? item[labelKey] + '' : '',
          size,
          color,
        }),
      }
      return new SMap.Marker(result)
    })
  }
  addMarkers(option: MarkerOption): unknown {
    const markers = this.transMarkerData(option)
    this.map.add(markers)
    return markers
  }
  remove(markers: unknown): void {
    this.map.remove(markers)
  }
}
