/**
 * 地图配置
 */
export type MapConfig<T> = T & {
  /** 地图容器（需要有宽高） */
  el: string
  /**
   * 在SMap中作为appKey使用
   * 在AIMap中作为accessToken使用
   */
  appKey: string
} & Partial<{
    /** 地图模式 */
    mode: '2D' | '3D'
    /** 地图初始中心点位 */
    center: Location
    /** 地图初始zoom等级 */
    zoom: number
    /** 地图zoom允许范围 */
    zooms: [number, number]
    /** 地图初始旋转角度 */
    bearing: number
    /** 地图初始仰角（3D用） */
    pitch: number
    /** 地图仰角允许范围 */
    pitchs: [number, number]
    /** 地图样式 */
    style: string
    showBuildingBlock: boolean
    rotateEnable: boolean
    family: string
  }>

export type SMapConfig = {
  /**
   * SMap 地图专用
   * @param internet 表示互联网2D
   * @param affairs 表示政务网2D
   * @param local3D 表示局域网3D
   * @param affairs3D 表示政务网3D
   * @param njdl 表示南京东路政务网3D
   */
  netType: 'internet' | 'affairs' | 'local3D' | 'affairs3D' | 'njdl'
}
export type AIMapConfig = {
  baseApiUrl: string
}

export type Location = [number, number] | [number, number, number]

/**
 * 坐标系类型
 * @param gcj02 高德坐标系
 * @param wgs84 gps坐标系
 * @param bd09 百度坐标系
 * @param cgcs2000 城建坐标系
 */
export type LocationType = 'gcj02' | 'wgs84' | 'bd09' | 'cgcs2000'

/**
 * 监听回调函数
 */
export type CallBack = (arg: any, oth?: any) => unknown

/** 经过封装的返回统一参数的点击回调 */
export type ClickCallBack = (
  arg: {
    /** 事件类型 */
    type: string
    /** 地图初始中心点位 */
    center: number[]
    /** 鼠标事件 */
    event: PointerEvent
    /** 触发时鼠标x位置 */
    clientX: number
    /** 触发时鼠标y位置 */
    clientY: number
    /** 鼠标点击x坐标 */
    x?: number
    /** 鼠标点击y坐标 */
    y?: number
    /** 地图触发事件的原始数据（即未经封装前的数据） */
    originData: any
  },
  oth?: any,
) => unknown

/**
 * 监听事件
 */
export interface Listeners {
  /** 加载完成触发 */
  load: (cb: CallBack) => unknown
  /** zoom变化触发 */
  zoom: (cb: CallBack) => unknown
  /** 地图移动 */
  move: (cb: CallBack) => unknown
  /** 失焦触发 */
  blur: (cb: CallBack) => unknown
  /** 聚焦触发 */
  focus: (cb: CallBack) => unknown
  /** 拖动触发 */
  drag: (cb: CallBack) => unknown
  /** 视图大小变化触发 */
  resize: (cb: CallBack) => unknown
  /** 点击触发 */
  click: (cb: ClickCallBack) => unknown
  /** 双击触发 */
  dblclick: (cb: CallBack) => unknown
  /** 滚轮触发 */
  mousewheel: (cb: CallBack) => unknown
}

type Position = 'top-left' | 'top-right' | 'bottom-left' | 'bottom-right'

/**
 * 控制器选项
 */
export interface ControlOptions {
  show?: boolean
  /** 控件位置 */
  position?: Position
  /** ScaleControl控件的最大长度，以像素为单位 */
  maxWidth?: number
  /** ScaleControl控件的距离单位 */
  unit?: 'imperial' | 'metric' | 'nautical'
}

/**
 * 控制器列表
 */
export interface Controls {
  home: (options?: ControlOptions) => void
  compass: (options?: ControlOptions) => void
  zoom: (options?: ControlOptions) => void
  scale: (options?: ControlOptions) => void
  fullScreen: (options?: ControlOptions) => void
  layerList: (options?: ControlOptions) => void
  measureLine: (options?: ControlOptions) => void
  measureArea: (options?: ControlOptions) => void
  basemapToggle: (options?: ControlOptions) => void
  underguroundSwitch: (options?: ControlOptions) => void
  bMapGallery: (options?: ControlOptions) => void
  bMapGalleryexpand: (options?: ControlOptions) => void
}

/**
 * zoom配置参数（维智地图用）
 */
export interface ZoomOptions {
  /** 如果 false ，则没有动画效果（默认true） */
  animate?: boolean
  /** 动态转换的持续时间，按毫秒计算（默认3000） */
  duration?: number
  /** 该函数持续的时间在 0~1 之间， 返回一个表示状态的数字，初始状态为 0，最终状态为 1 */
  easing?: (t?: number) => number | void
  /** 动态转换结束后，目标中心与实际地图容器中心间的偏差，单位为像素 */
  offset?: [number, number]
}

/**
 * focus配置参数（测绘院地图只有zoom有用）
 */
export interface FocusOptions {
  /**地图缩放等级（默认当前等级） */
  zoom?: number
  /** 地图倾斜角度（默认当前角度） */
  pitch?: number
  /** 地图旋转角度（默认当前角度） */
  bearing?: number
  /** 图层距离容器四周距离(单位像素) */
  padding?: { top?: number; bottom?: number; left?: number; right?: number }
  /** 动画速度（默认1.2） */
  speed?: number
  /** 默认1.42 */
  curve?: number
  /** 整个fly动画持续毫秒时间（默认3000） */
  maxDuration?: number
}

/**
 * 覆盖物列表
 */
export interface Layers {
  point: (data: LayerOption) => unknown
  line: (data: LayerOption) => unknown
  polygon: (data: LayerOption) => unknown
}

/** 覆盖物点位数据 */
export interface PointsData {
  id?: string
  x?: number
  y?: number
  z?: number
  X?: number
  Y?: number
  Z?: number
  gpsx?: number
  gpsy?: number
  gpsz?: number
  icon?: string
  spatialReference?: LocationType
  path?: [number, number][]
  [key: string]: any
}

/**
 * 覆盖物选项
 */
export interface LayerOption {
  data: PointsData | PointsData[]
  size?: number | [number, number]
  icon?: string
  spatialReference?: LocationType
  labelKey?: string
  labelOptions?: {
    size?: number
    /** [水平偏移量，垂直偏移量] */
    offset?: [number, number] | [number, number, number]
    color?: string
  }
  color?: string
  fillColor?: string
  lineCap?: string
  lineStyle?: string
  lineJoin?: 'bevel' | 'round' | 'miter'
  lineWidth?: number
  _key?: string
}
/** 边界选项 */
export interface BoundaryOption {
  /** 根据type模糊匹配名称  */
  name: string
  /**
   *  边界类型
   *
   * 'jd_boundary' : 街道
   *
   * 'jwh_boundary' : 南东工作站
   */
  type?: string
  weight?: number
  count?: number
  color?: string
  maskColor?: number[]
}
