import { LayerOption, PointsData } from '../types'

/** 转化点状覆盖物 */
export function toPoints(options: LayerOption) {
  const { data, size = 16, icon, labelKey, labelOptions, _key } = options
  const transData = (item: PointsData) => {
    const iconSize = Array.isArray(size)
      ? new SMap.Size(...size)
      : new SMap.Size(size, size)
    const x = item.minx || item.x || item.maxx || item.X || item.gpsx
    const y = item.miny || item.y || item.maxy || item.Y || item.gpsy
    if ((x !== 0 && !x) || (y !== 0 && !y)) {
      console.error(`add point: 非法的坐标[${x}, ${y}] 存在于数据: `, item)
    }
    const z = item.Z || item.z || item.gpsz || 0
    const result: any = {
      icon: {
        size: iconSize,
        image: item.icon || icon,
      },
      attributes: { ...item, _key },
      position: z ? [x, y, z] : [x, y],
    }
    if (labelKey) {
      result.label = new SMap.Label({
        text: item[labelKey] + '',
        size: labelOptions?.size,
        color: labelOptions?.color,
        xoffset: labelOptions?.offset?.[0],
        yoffset: labelOptions?.offset?.[1],
        zoffset: labelOptions?.offset?.[2],
        verticalAlignment: 'middle',
        horizontalAlignment: 'center',
      })
    }
    return new SMap.Marker(result)
  }
  if (Array.isArray(data)) {
    return data.map(transData)
  } else {
    return [transData(data)]
  }
}

/** 转化线状覆盖物 */
export function toLines(options: LayerOption) {
  const {
    data,
    color = 'blue',
    labelKey,
    labelOptions,
    lineCap = 'square',
    lineStyle = 'solid',
    lineJoin = 'round',
    lineWidth = 1,
  } = options
  const transData = (item: PointsData) => {
    const result: any = {
      path: item.path?.map(([x, y]) => new SMap.LngLat(x, y)),
      attributes: { ...item },
      cap: lineCap,
      strokeColor: color,
      style: lineStyle,
      lineJoin: lineJoin,
      lineWidth, // 测绘院地图貌似没法设置线段粗细
    }
    if (labelKey) {
      result.label = new SMap.Label({
        text: item[labelKey] + '',
        size: labelOptions?.size,
        color: labelOptions?.color,
        xoffset: labelOptions?.offset?.[0],
        yoffset: labelOptions?.offset?.[1],
        verticalAlignment: 'middle',
        horizontalAlignment: 'center',
      })
    }
    return new SMap.Polyline(result)
  }
  if (Array.isArray(data)) {
    return data.map(transData)
  } else {
    return [transData(data)]
  }
}

/** 转化面状覆盖物 */
export function toPolygons(options: LayerOption) {
  const {
    data,
    color = 'blue',
    fillColor = 'rgba(135,206,235,0.5)',
    labelKey,
    labelOptions,
    lineCap = 'square',
    lineStyle = 'solid',
    lineJoin = 'round',
    lineWidth = 3,
  } = options

  const transData = (item: PointsData) => {
    const result: any = {
      paths: item.path?.map(([x, y]) => new SMap.LngLat(x, y)),
      attributes: { ...item },
      cap: lineCap,
      strokeColor: color,
      fillColor,
      strokeWeight: lineWidth,
      style: lineStyle,
      lineJoin: lineJoin,
    }
    if (labelKey) {
      result.label = new SMap.Label({
        text: item[labelKey] + '',
        size: labelOptions?.size,
        color: labelOptions?.color,
        xoffset: labelOptions?.offset?.[0],
        yoffset: labelOptions?.offset?.[1],
        verticalAlignment: 'middle',
        horizontalAlignment: 'center',
      })
    }
    return new SMap.Polygon(result)
  }
  if (Array.isArray(data)) {
    return data.map(transData)
  } else {
    return [transData(data)]
  }
}
