import axios from 'axios'
import qs from 'qs'
import api from './api'
import Store from '@/store'
import Cookie from '@/util/local-cookie'
import router from '@/router'
import Common from '@/util/common'
import { message } from 'ant-design-vue'

// 配置请求的根域名和超时时间
const Axios = axios.create({
  baseURL: api.BASE_URL,
  timeout: 15000,
})
const CancelToken = axios.CancelToken
let cancelRequest = null

// 处理请求状态码
const reponseCodeHandler = (res) => {
  const code = res.data && res.data.code
  if (code && code != '200') {
    switch (code) {
    case '911':
      const params = {
        grant_type: 'refresh_token',
        client_id: 'house',
        client_secret: 'house',
        refreshToken: Cookie.get('refresh_token'),
      }
      request({
        method: 'POST',
        url: api.REFRESH_TOKEN_POST,
        params,
        contentType: 'application/x-www-form-urlencoded;charset=UTF-8',
      }).then(res => {
        const access_token = Common.confirm(res, 'data.content.access_token')
        const refresh_token = Common.confirm(res, 'data.content.refresh_token')
        Cookie.set('token', access_token)
        Cookie.set('refresh_token', refresh_token)
      })
      break
    default:
      message.error(code)
      break
    }
  }
}

// 根据报错的状态码进行错误处理
const errorHandler = (err) => {
  const errStatus = (err.response && err.response.status) || (err.data && err.data.errcode)
  message.error(errStatus)
}

Axios.interceptors.request.use(config => {
  const token = Cookie.get('token')
  if (token) {
    config.headers.Authorization = token
  }
  return config
}, error => {
  return Promise.reject(error)
})

Axios.interceptors.response.use(response => {
  reponseCodeHandler(response)
  return response.data
}, error => {
  errorHandler(error)
  return error.response
})

/**
 * 请求
 * @param {String} method [请求方法]
 * @param {String} url [请求地址]
 * @param {Object} params [请求参数]
 * @param {String} contentType [请求头，默认为'application/json;charset=UTF-8']
 * @param {Boolean} hideLoading [隐藏请求时的loading图，默认为false]
 */
const request = ({ method, url, params, contentType = 'application/json;charset=UTF-8', hideLoading = false }) => {
  if (!url || typeof(url) != 'string') {
    throw new Error('接口URL不正确')
  }
  if (!params || typeof(params) == 'string' || typeof(params) == 'number') {
    params = {}
  }
  let config = {
    method,
    url,
    headers: {
      'X-Requested-With': 'XMLHttpRequest',
      'Content-Type': contentType,
    },
    cancelToken: new CancelToken((c) => {
      cancelRequest = c
    }),
  }
  if (method === 'GET') {
    config = Object.assign(config, { params })
  } else {
    if (contentType.toLowerCase().indexOf('x-www-form-urlencoded') >= 0) {
      config = Object.assign(config, { data: qs.stringify(params) })
    } else {
      config = Object.assign(config, { data: params })
    }
  }
  if (!hideLoading) {
    Store.commit('SET_LOADING', true)
  }

  return new Promise((resolve, reject) => {
    Axios(config)
      .then(res => {
        resolve(res)
        Store.commit('SET_LOADING', false)
      }).catch(err => {
        reject(err)
        Store.commit('SET_LOADING', false)
      })
  })
}

export default {
  /**
     * 取消请求
     * @param {String} txt [取消请求时需要显示在控制台的提示信息]
     */
  cancel(txt = '取消请求') {
    Store.commit('SET_LOADING', false)
    if (typeof(cancelRequest) === 'function') {
      return cancelRequest(txt)
    }
  },
  get(args) {
    return request({ method: 'GET', ...args })
  },
  post(args) {
    return request({ method: 'POST', ...args })
  },
  put(args) {
    return request({ method: 'PUT', ...args })
  },
  delete(args) {
    return request({ method: 'DELETE', ...args })
  },
  all(...ajaxs) {
    return Promise.all(ajaxs)
  },
}
