import { ajax, api } from '@/ajax'
import { Dispatch, Commit } from './index'
import dayjs from 'dayjs'
import yi from '@/assets/images/yi.png'
import shi from '@/assets/images/shi.png'
import zhu from '@/assets/images/zhu.png'
import xing from '@/assets/images/xing.png'
import other from '@/assets/images/other.png'
import smStar from '@/assets/images/star-sm.png'
import star from '@/assets/images/star.png'

interface Method {
  commit: Commit
  dispatch: Dispatch
}

const today = dayjs().format('YYYY-MM-DD')

export default {
  initData({ dispatch }: Method): void {
    /** 初始化数据 */
    dispatch('getPropertySummary')
    dispatch('GET_PUBLICSAFE_COMMANDSYSTEM')
    dispatch('GET_PUBLICSAFE_STREETPOWER')
    dispatch('GET_PUBLICSAFE_PARTY')
    dispatch('GET_PUBLICSAFE_ECONOMIC')
    dispatch('GET_PUBLICSAFE_TRUE_COMPANY')
    dispatch('GET_YOUSUBIDA_RANK')
    dispatch('GET_YOUSUBIDA_CLASSIFICATION_RANK')
    dispatch('GET_URGENTCASE', {
      bxmaintainType: '1',
      // status_in: '待派遣,已派单',
      acceptTime_gte: today,
      pageSize: 100,
    })
    dispatch('GET_TODAYCASE')
    dispatch('GET_STAGECASE')
    dispatch('GET_WORKSTATION')
    dispatch('GET_TRACKEVENT', {
      // status_in: '待处置,已签收',
      acceptTime_gte: today,
      pageSize: 100,
    })
    dispatch('GET_CASE_CLASSIFICATION_RANK', {
      managementType: '综合',
    })
    dispatch('GET_MOREHAPPEN_ADDR', {
      managementType: '综合',
    })
    dispatch('GET_RESPONSIVITY')
    dispatch('GET_ENTRY')
    dispatch('GET_PUBLICSAFE_SOS_MATERIAL')
    dispatch('GET_PUBLICSAFE_OUTHOUSEWALL')
    dispatch('GET_PUBLICSAFE_BUILDINGS')
    dispatch('GET_PUBLICSAFE_PIEDATA')
    dispatch('GET_PUBLICSAFE_HOUSE')
    dispatch('GET_PUBLICSAFE_NONRESIDENT')
    dispatch('GET_PUBLICSAFE_RIGHT_GATE')
    dispatch('GET_PUBLICSAFE_RIGHT_SAFEMAP')
    dispatch('GET_PUBLICSAFE_NEWFIND')
    dispatch('getEcologyList')
    dispatch('getEcologyPoints')
    dispatch('getEcologyConvenient')
    dispatch('getEcologyPhoto')
    dispatch('getElevCount')
    dispatch('GET_BEAUTY_STREET')
    dispatch('GET_BEAUTY_HOME')
    dispatch('GET_FACILITY_CONVENIENT')
  },

  // 物业三架马车数据 （缺交叉任职数）
  async getPropertySummary({ commit }: Method): Promise<void> {
    const {
      GET_COMMITTEE: url1,
      GET_INDUSTRY_AUTHORITY: url2,
      GET_COMPANY: url3,
    } = api
    const [num1, num2, num3] = await Promise.all(
      [url1, url2, url3].map(async (url) => {
        return (await ajax.get({ url, params: { pageSize: 1 } })).data.totalRows
      })
    )
    commit('SET_PROPERTY_SUMMARY', { num1: +num1, num2: +num2, num3: +num3 })
  },

  // 公共服务 - 门岗值守 - 社会物业和直管公房数据
  async GET_PUBLICSAFE_NEWFIND({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_PUBLICSAFE_NEWFIND })
    ).data
    const obj = {
      done: 0,
      doing: 0,
      find: 0,
      newFind: content.newDiscovery,
    }
    function format(params: any) {
      params.forEach((item: any) => {
        obj.find += item.num
        switch (item.status) {
          case '已结案':
            obj.done = item.num
            break
          case '待处置':
            obj.find = item.num
            break
          default:
            break
        }
      })
    }
    format(content.caseNum)

    commit('SET_PUBLICSAFE_NEWFIND', obj)
  },

  // 公共服务 - 门岗值守 - 社会物业和直管公房数据
  async GET_PUBLICSAFE_RIGHT_GATE({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_PUBLICSAFE_RIGHT_GATE })
    ).data
    const shehuiwuye: any[] = []
    const publicHouse: any[] = []
    function formatData(arr: any[], obj: { [x: string]: any }) {
      for (const k in obj) {
        switch (k) {
          case 'isWatch':
            arr.push({
              index: 3,
              name: '值守处',
              value: obj[k] || 0,
            })
            break
          case 'isClosed':
            arr.push({
              index: 1,
              name: '已封闭',
              value: obj[k] || 0,
            })
            break
          case 'addrNum':
            arr.push({
              index: 0,
              name: '出入口',
              value: obj[k] || 0,
            })
            break
          case 'isSentry':
            arr.push({
              index: 2,
              name: '有门岗处',
              value: obj[k] || 0,
            })
            break

          default:
            break
        }
      }
      arr.sort((a, b) => {
        return a.index - b.index
      })
    }
    formatData(shehuiwuye, content.property)
    formatData(publicHouse, content.publicHouse)
    commit('SET_PUBLICSAFE_GATESENTRY_SHEHUIWUYE', shehuiwuye)
    commit('SET_PUBLICSAFE_GATESENTRY_ZHIGUANGONGFANG', publicHouse)
  },

  // 公共安全 - 门岗值守 - 平安地图
  async GET_PUBLICSAFE_RIGHT_SAFEMAP({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_PUBLICSAFE_RIGHT_SAFEMAP,
        params: { pageSize: 1000 },
      })
    ).data
    commit('SET_PUBLICSAFE_SAFEMAP', content)
  },

  // 公共服务 - 实有房屋
  async GET_PUBLICSAFE_HOUSE({ commit }: Method): Promise<void> {
    const { content } = (await ajax.get({ url: api.GET_PUBLICSAFE_HOUSE })).data
    commit('SET_PUBLICSAFE_HOUSE', content)
    const obj = {
      spf: 0,
      other: 0,
      zggf: 0,
    }
    content.stKind.forEach((item: { stKindName: string; totHous: number }) => {
      switch (item.stKindName) {
        case '商品房':
          obj.spf = item.totHous
          break
        case '直管公房':
          obj.zggf = item.totHous
          break
        default:
          obj.other += item.totHous
          break
      }
    })
    commit('SET_PUBLICSAFE_HOUSE_STKIND', content.stKind)
    commit('SET_PUBLICSAFE_HOUSE_STKIND', obj)
  },

  // 公共服务 - 非居房屋
  async GET_PUBLICSAFE_NONRESIDENT({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_PUBLICSAFE_NONRESIDENT })
    ).data
    commit('SET_PUBLICSAFE_NONRESIDENT', content)
  },

  // 公共服务 - 实有单位
  async GET_PUBLICSAFE_TRUE_COMPANY({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_PUBLICSAFE_TRUE_COMPANY })
    ).data
    commit('SET_PUBLICSAFE_TRUE_COMPANY', content)
  },

  // 公共服务 - 应急物资储备
  async GET_PUBLICSAFE_SOS_MATERIAL({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_PUBLICSAFE_SOS_MATERIAL })
    ).data
    commit('SET_PUBLICSAFE_SOS_MATERIAL', content)
  },

  // 公共服务 - 房屋外立面
  async GET_PUBLICSAFE_OUTHOUSEWALL({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_PUBLICSAFE_OUTHOUSEWALL })
    ).data
    commit('SET_PUBLICSAFE_OUTHOUSEWALL', content)
  },

  // 公共安全 - 重点关注楼宇
  async GET_PUBLICSAFE_BUILDINGS({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_PUBLICSAFE_BUILDINGS,
        params: { pageSize: 1000 },
      })
    ).data
    commit('SET_PUBLICSAFE_BUILDINGS', content)
  },

  // 公共服务 - 房屋外立面饼图
  async GET_PUBLICSAFE_PIEDATA({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_PUBLICSAFE_PIEDATA })
    ).data
    commit('SET_PUBLICSAFE_PIEDATA', content)
  },

  /** 公共服务 - 生态 */
  async getEcologyList({ commit }: Method): Promise<void> {
    const { content } = (await ajax.get({ url: api.GET_ECOLOGY_LIST })).data
    commit('SET_ECOLOGY_LIST', content)
  },
  async getEcologyPoints({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_ECOLOGY_POINTS,
        params: { pageSize: 1000 },
      })
    ).data
    const result = [
      ...(content || []).map((item: any) => {
        switch (item.labelName) {
          case '衣':
            item.icon = yi
            item._type = 'yi'
            break
          case '食':
            item.icon = shi
            item._type = 'shi'
            break
          case '住':
            item.icon = zhu
            item._type = 'zhu'
            break
          case '行':
            item.icon = xing
            item._type = 'xing'
            break
          // case '医':
          //   item.icon = yi2
          //   item._type = 'other'
          //   break
          // case '学':
          //   item.icon = xue
          //   item._type = 'other'
          //   break
          // case '商':
          //   item.icon = shang
          //   item._type = 'other'
          //   break
          // case '政':
          //   item.icon = zheng
          //   item._type = 'other'
          //   break
          default:
            item.icon = other
            item._type = 'other'
            break
        }
        item.size = [30, 35]
        if (item.stAddr === '大沽路123号') {
          item.icon = smStar
          item.size = 20
          item._type = 'govern'
        }
        if (item.stAddr === '人民大道200号') {
          item.icon = star
          item.size = 20
          item._type = 'govern'
        }
        return item
      }),
    ]
    commit('SET_ECOLOGY_POINTS', result)
  },
  async getEcologyConvenient({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_ECOLOGY_CONVENIENT })
    ).data
    commit('SET_ECOLOGY_CONVENIENT', content)
  },
  async getEcologyPhoto({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_ECOLOGY_PHOTO,
        params: { pageNo: 1, pageSize: 1000 },
      })
    ).data
    commit('SET_ECOLOGY_PHOTO', content)
  },
  async getElevCount({ commit }: Method): Promise<void> {
    const { content } = (await ajax.get({ url: api.GET_ELEV_COUNT })).data
    commit('SET_ELEV_COUNT', content)
  },

  // 左侧公共 - 指挥体系
  async GET_PUBLICSAFE_COMMANDSYSTEM({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_PUBLICSAFE_COMMANDSYSTEM })
    ).data
    const { commander = '', deputyCommander = '' } =
      (await ajax.get({ url: api.GET_PUBLICSAFE_MASTER }))?.data?.content || {}

    commit('SET_PUBLICSAFE_COMMANDSYSTEM', {
      commander,
      deputyCommander,
      list: content || [],
    })
  },

  // 左侧公共 - 网格工作站
  async GET_PUBLICSAFE_STREETPOWER({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_PUBLICSAFE_STREETPOWER })
    ).data
    const { content: gridPowerList } = (
      await ajax.get({ url: api.GET_GRIDPOWER, params: { pageSize: 1000 } })
    ).data
    content.forEach((item: any) => {
      item.gridPower =
        gridPowerList.filter((e: any) => e.workstationId == item.id) || []
    })
    commit('SET_PUBLICSAFE_STREETPOWER', content || [])
  },

  // 左侧公共 - 党建
  async GET_PUBLICSAFE_PARTY({ commit }: Method): Promise<void> {
    const { content } = (await ajax.get({ url: api.GET_PUBLICSAFE_PARTY })).data
    commit('SET_PUBLICSAFE_PARTY', content)
  },

  // 左侧公共 - 经济
  async GET_PUBLICSAFE_ECONOMIC({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_PUBLICSAFE_ECONOMIC })
    ).data
    commit('SET_PUBLICSAFE_ECONOMIC', content)
  },

  /** 小区屏 */
  initCommunityData({ dispatch }: Method, id: string): void {
    dispatch('getCommunityInfo', id)
    dispatch('getIssueList', id)
    dispatch('getScoreIssue', id)
    dispatch('getCombineList', id)
  },
  async getCommunityInfo({ commit }: Method, id: string): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_COMMUNITY_INFO,
      })
    ).data
    commit(
      'SET_COMMUNITY_INFO',
      content.find((item: any) => item.id === id)
    )
  },
  async getIssueList({ commit }: Method, id: string): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_ISSUE_LIST,
        params: { bigscreenCommunityId: id },
      })
    ).data
    commit('SET_ISSUE_DATA', content)
  },
  async getScoreIssue({ commit }: Method, id: string): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_SCORE_ISSUE,
        params: { bigscreenCommunityId: id },
      })
    ).data
    commit('SET_SCORE_ISSUE', content)
  },
  async getCombineList({ commit }: Method, id: string): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_COMBINE_LIST,
        params: { bigscreenCommunityId: id, pageSize: 200 },
      })
    ).data
    commit('SET_COMBINE_DATA', content)
  },

  /** 工作站 */
  initStationData({ dispatch }: Method, id: string): void {
    if (!id) return
    dispatch('getStationById', id)
    dispatch('getDutySum', id)
    dispatch('getOldService', id)
    dispatch('getLifeConvenient', id)
  },
  async getStationById({ commit }: Method, id: string): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_STATION, params: { id } })
    ).data
    commit('SET_STATION', content && content[0])
  },
  async getDutySum({ commit }: Method, gridId: string): Promise<void> {
    const { content } = (
      await ajax.get({ url: api.GET_DUTY_SUM, params: { gridId } })
    ).data
    commit('SET_DUTY_SUM', content)
  },
  async getDutyList(
    { commit }: Method,
    params: { [key: string]: string | number }
  ): Promise<void> {
    if (!params.gridId) return
    const { content } = (
      await ajax.get({
        url: api.GET_DUTY_LIST,
        params,
      })
    ).data
    commit('SET_DUTY_LIST', content)
  },
  async getDutyRank(
    { commit }: Method,
    params: { [key: string]: string }
  ): Promise<void> {
    if (!params.gridId) return
    const { content } = (
      await ajax.get({
        url: api.GET_DUTY_RANK,
        params,
      })
    ).data
    commit('SET_DUTY_RANK', content)
  },
  async getDutyEvents(
    { commit }: Method,
    params: { [key: string]: string }
  ): Promise<void> {
    if (!params.gridId) return
    const { content } = (
      await ajax.get({
        url: api.GET_DUTY_EVENTS,
        params,
      })
    ).data
    commit('SET_DUTY_EVENTS', content)
  },
  async getDutyPlaces(
    { commit }: Method,
    params: { [key: string]: string }
  ): Promise<void> {
    if (!params.gridId) return
    const { content } = (
      await ajax.get({
        url: api.GET_DUTY_PLACES,
        params,
      })
    ).data
    commit('SET_DUTY_PLACES', content)
  },
  async getDutyRanking(
    { commit }: Method,
    params: { [key: string]: string }
  ): Promise<void> {
    if (!params.gridId) return
    const { content } = (
      await ajax.get({
        url: api.GET_DUTY_RANKING,
        params,
      })
    ).data
    commit('SET_DUTY_RANKING', content)
  },
  async getOldService({ commit }: Method, id: string): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_OLD_SERVICE,
        params: {
          workstationId: id,
        },
      })
    ).data
    commit('SET_OLD_SERVICE', content)
  },
  async getLifeConvenient({ commit }: Method, id: string): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_LIFE_CONVENIENT,
        params: {
          workstationId: id,
        },
      })
    ).data
    commit('SET_LIFE_CONVENIENT', content)
  },

  /** 公共管理 */
  // 业务入口
  async GET_ENTRY({ commit }: Method): Promise<void> {
    const { content } = (await ajax.get({ url: api.GET_ENTRY })).data
    commit('SET_ENTRY', content)
  },

  async GET_YOUSUBIDA_RANK({ commit }: Method): Promise<void> {
    // 有诉必答 - 占比
    const { content } = (await ajax.get({ url: api.GET_YOUSUBIDA_RANK })).data
    commit('SET_YOUSUBIDA_RANK', content)
  },
  async GET_YOUSUBIDA_CLASSIFICATION_RANK({ commit }: Method): Promise<void> {
    // 有诉必答 - 分类排行
    const { content } = (
      await ajax.get({
        url: api.GET_YOUSUBIDA_CLASSIFICATION_RANK,
      })
    ).data
    commit('SET_YOUSUBIDA_CLASSIFICATION_RANK', content)
  },
  async GET_URGENTCASE(
    { commit }: Method,
    params: {
      [key: string]: unknown
    }
  ): Promise<void> {
    // 有诉必答 - 紧急案件
    const { content } = (
      await ajax.get({
        url: api.GET_URGENTCASE,
        params,
      })
    ).data
    commit('SET_URGENTCASE', content)
  },
  async GET_TODAYCASE(
    { commit }: Method,
    params: {
      [key: string]: string
    }
  ): Promise<void> {
    // 有求必应 - 今日案件数
    const { content } = (
      await ajax.get({
        url: api.GET_TODAYCASE,
        params,
      })
    ).data
    commit('SET_TODAYCASE', content)
  },
  async GET_STAGECASE(
    { commit }: Method,
    params: {
      [key: string]: string
    }
  ): Promise<void> {
    // 有求必应 - 各阶段案件数
    const { content } = (
      await ajax.get({
        url: api.GET_STAGECASE,
        params,
      })
    ).data
    commit('SET_STAGECASE', content)
  },
  async GET_WORKSTATION(
    { commit }: Method,
    params: {
      [key: string]: string
    }
  ): Promise<void> {
    // 有求必应 - 工作站分类
    const { content } = (
      await ajax.get({
        url: api.GET_WORKSTATION,
        params,
      })
    ).data
    commit('SET_WORKSTATION', content)
    commit('SET_WORKSTATION_PIEDATA', content[0].statistical)
  },
  async GET_TRACKEVENT(
    { commit }: Method,
    params: {
      [key: string]: unknown
    }
  ): Promise<void> {
    // 有求必应 - 事件跟踪
    const { content } = (
      await ajax.get({
        url: api.GET_TRACKEVENT,
        params,
      })
    ).data
    commit('SET_TRACKEVENT', content)
  },
  async GET_CASE_CLASSIFICATION_RANK(
    { commit }: Method,
    params: {
      [key: string]: unknown
    }
  ): Promise<void> {
    // 案件分析 - 分类排行
    const { content } = (
      await ajax.get({
        url: api.GET_CASE_CLASSIFICATION_RANK,
        params,
      })
    ).data
    commit('SET_CASE_CLASSIFICATION_RANK', content)
  },
  async GET_MOREHAPPEN_ADDR(
    { commit }: Method,
    params: {
      [key: string]: string
    }
  ): Promise<void> {
    // 案件分析 - 多发地点
    const { content } = (
      await ajax.get({
        url: api.GET_MOREHAPPEN_ADDR,
        params,
      })
    ).data
    commit('SET_MOREHAPPEN_ADDR', content)
  },

  //响应度
  async GET_RESPONSIVITY({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_RESPONSIVITY,
      })
    ).data
    commit('SET_RESPONSIVITY', content)
  },

  async GET_BEAUTY_STREET({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_BEAUTY_STREET,
        params: { pageSize: 10000 },
      })
    ).data
    commit('SET_BEAUTY_STREET', content)
  },
  async GET_BEAUTY_HOME({ commit }: Method): Promise<void> {
    const { content } = (
      await ajax.get({
        url: api.GET_BEAUTY_HOME,
        params: { pageSize: 10000 },
      })
    ).data
    commit('SET_BEAUTY_HOME', content)
  },

  async GET_FACILITY_CONVENIENT({ commit }: Method): Promise<void> {
    const result = {
      为老:
        (
          await ajax.get({
            url: api.GET_FOR_OLD,
            params: { pageSize: 10000 },
          })
        )?.data?.content || [],
      菜场:
        (
          await ajax.get({
            url: api.GET_MARKETS,
            params: { labelName: '菜场', pageSize: 10000 },
          })
        )?.data?.content || [],
      便利店:
        (
          await ajax.get({
            url: api.GET_MARKETS,
            params: { labelName: '便利店', pageSize: 10000 },
          })
        )?.data?.content || [],
      电话亭:
        (
          await ajax.get({
            url: api.GET_MARKETS,
            params: { labelName: '电话亭', pageSize: 10000 },
          })
        )?.data?.content || [],
      餐车:
        (
          await ajax.get({
            url: api.GET_MARKETS,
            params: { labelName: '餐车', pageSize: 10000 },
          })
        )?.data?.content || [],
      公厕:
        (
          await ajax.get({
            url: api.GET_MARKETS,
            params: { labelName: '公厕', pageSize: 10000 },
          })
        )?.data?.content || [],
      街面废物箱:
        (
          await ajax.get({
            url: api.GET_MARKETS,
            params: { labelName: '街面废物箱', pageSize: 10000 },
          })
        )?.data?.content || [],
      垃圾箱房:
        (
          await ajax.get({
            url: api.GET_MARKETS,
            params: { labelName: '垃圾箱房', pageSize: 10000 },
          })
        )?.data?.content || [],
      药房:
        (
          await ajax.get({
            url: api.GET_MARKETS,
            params: { labelName: '药房', pageSize: 10000 },
          })
        )?.data?.content || [],
    }
    commit('SET_FACILITY_CONVENIENT', result)
  },
}
